import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Globe, Languages, BookOpen, Award, ArrowLeft } from 'lucide-react'
import { motion } from 'framer-motion'
import './EnrollCourse.css'

const EnrollCourse = () => {
  const navigate = useNavigate()
  const [selectedLanguage, setSelectedLanguage] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const languages = [
    { 
      id: 'english', 
      name: 'Английский', 
      iconType: 'globe',
      iconComponent: Globe,
      description: 'Самый популярный язык в мире'
    },
    { 
      id: 'spanish', 
      name: 'Испанский', 
      iconType: 'languages',
      iconComponent: Languages,
      description: 'Язык страсти и культуры'
    },
    { 
      id: 'chinese', 
      name: 'Китайский', 
      iconType: 'book',
      iconComponent: BookOpen,
      description: 'Откройте двери в Азию'
    },
    { 
      id: 'german', 
      name: 'Немецкий', 
      iconType: 'award',
      iconComponent: Award,
      description: 'Язык бизнеса и науки'
    }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedLanguage && selectedDate) {
      const savedCourses = localStorage.getItem('languageCourses')
      const courses = savedCourses ? JSON.parse(savedCourses) : []
      
      const languageData = languages.find(l => l.id === selectedLanguage)
      const newCourse = {
        id: Date.now(),
        language: languageData.name,
        level: 'Beginner',
        progress: 0,
        rating: 0,
        iconType: languageData.iconType
      }

      courses.push(newCourse)
      localStorage.setItem('languageCourses', JSON.stringify(courses))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedLanguage('')
        setSelectedDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="enroll-minimal-page">
      <motion.div 
        className="enroll-header-minimal"
        initial={{ opacity: 0, y: -20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.5 }}
      >
        <button 
          className="back-button"
          onClick={() => navigate('/')}
        >
          <ArrowLeft size={20} strokeWidth={2} />
        </button>
        <div>
          <h1 className="enroll-title-minimal">Записаться на курс</h1>
          <p className="enroll-subtitle-minimal">Выберите язык и дату старта</p>
        </div>
      </motion.div>

      <motion.div 
        className="enroll-content-wrapper"
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        transition={{ duration: 0.6, delay: 0.2 }}
      >
        <form className="enroll-form-minimal" onSubmit={handleSubmit}>
          <div className="form-section-minimal">
            <label className="form-label-minimal">
              <Globe size={18} strokeWidth={2} />
              Язык
            </label>
            <div className="languages-grid-minimal">
              {languages.map(language => (
                <motion.button
                  key={language.id}
                  type="button"
                  className={`language-card-minimal ${selectedLanguage === language.id ? 'selected' : ''}`}
                  onClick={() => setSelectedLanguage(language.id)}
                  whileHover={{ y: -2 }}
                  whileTap={{ scale: 0.98 }}
                >
                  <div className="card-minimal-icon">
                    <language.iconComponent size={32} strokeWidth={2} />
                  </div>
                  <div className="card-minimal-content">
                    <h3 className="card-minimal-name">{language.name}</h3>
                    <p className="card-minimal-desc">{language.description}</p>
                  </div>
                  {selectedLanguage === language.id && (
                    <motion.div 
                      className="card-minimal-check"
                      initial={{ scale: 0 }}
                      animate={{ scale: 1 }}
                    >
                      <CheckCircle size={20} strokeWidth={2} />
                    </motion.div>
                  )}
                </motion.button>
              ))}
            </div>
          </div>

          <div className="form-section-minimal">
            <label htmlFor="date" className="form-label-minimal">
              <Calendar size={18} strokeWidth={2} />
              Дата старта
            </label>
            <input
              type="date"
              id="date"
              className="input-minimal"
              value={selectedDate}
              onChange={(e) => setSelectedDate(e.target.value)}
              min={minDate}
              max={maxDate}
              required
            />
          </div>

          <motion.button 
            type="submit" 
            className="submit-minimal" 
            disabled={!selectedLanguage || !selectedDate}
            whileHover={{ y: -2 }}
            whileTap={{ scale: 0.98 }}
          >
            Записаться
          </motion.button>

          {isSubmitted && (
            <motion.div 
              className="success-minimal"
              initial={{ opacity: 0, y: 10 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.3 }}
            >
              <CheckCircle size={20} strokeWidth={2} />
              <span>Вы успешно записались на курс!</span>
            </motion.div>
          )}
        </form>
      </motion.div>
    </div>
  )
}

export default EnrollCourse
